<?php
declare(strict_types=1);

/*
 * Author: Ricardo Vega Jr. - www.noctusoft.com
 * Copyright (C) 2011 Ricardo Vega Jr.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

define('ROOT_PATH', dirname(__FILE__) . '/');
define('DUMP_FILENAME', ROOT_PATH . 'logs/gamedump.txt');
define('DUMP_SORTED_FILENAME', ROOT_PATH . 'logs/gamedump_sorted.txt');
define('SCHED_FILENAME', ROOT_PATH . 'logs/schdump.txt');
define('ROUND_ROBIN', -1);
define('DOUBLE_ROUND_ROBIN', -2);
define('RAW_GAME_ARRAY_COUNT', -99999);

date_default_timezone_set('America/Chicago');

function GenerateSchedule(string $EntityList, int $GamesPerEntity, int $EntityPerGame): array
{
    $sch = new Schedule();
    $arrEntityList = explode(',', $EntityList);
    $sch->setGamesForEachEntity($GamesPerEntity);
    $sch->setEntityCount(count($arrEntityList));
    $sch->setEntitiesPerGame($EntityPerGame);
    $ret = $sch->generateSchedule();
    shuffle($arrEntityList);
    for ($iGame = 0; $iGame < count($ret); $iGame++) {
        for ($iTeamSlot = 0; $iTeamSlot < count($ret[$iGame]); $iTeamSlot++) {
            $ret[$iGame][$iTeamSlot] = trim($arrEntityList[$ret[$iGame][$iTeamSlot] - 1]);
        }
    }
    return $ret;
}

class Schedule
{
    private ?string $error = null;
    private ?string $entity = null; // Team/Player
    private int $entityCount = 18; // The amount of Team/Player Participating 
    private int $entitiesPerGame = 3; // The amount of Team/Players per Game
    private int $subGamePerGame = 1; // The amount of subgames per game (this effectively multiplies the amount of entities in a game)
    private int $gamesForEachEntity = 6;
    private array $gamesPlayed = [];
    private array $rawGames = [];
    private array $schedule = [];

    public function __construct()
    {
    }

    private function init(): void
    {
        $this->gamesPlayed = array_fill(0, $this->entityCount - 1, 0);
    }

    public function generateSchedule(): array
    {
        $this->init();
        $this->createRawGames();
        $this->schedule = $this->calcRawSchedule();
        $this->schedule = $this->evenlyDistEntPos($this->schedule);
        return $this->schedule;
    }

    private function calcDistFromLastPlayed(array $currentSchedule, array $game, array $vData = [], int $teamSlot = 0): array
    {
        if ($teamSlot > $this->entitiesPerGame - 1) {
            return $vData;
        }

        $entityToSearchFor = $game[$teamSlot];
        $dist = 0;
        $vData[$teamSlot] = count($this->rawGames);

        for ($iGame = count($currentSchedule) - 1; $iGame >= 0; $iGame--) {
            $dist++;
            if (in_array($entityToSearchFor, $currentSchedule[$iGame])) {
                $vData[$teamSlot] = $dist;
                break;
            }
        }

        return $this->calcDistFromLastPlayed($currentSchedule, $game, $vData, $teamSlot + 1);
    }

    private function evenlyDistEntPos(array $currentSchedule): array
    {
        $currentScheduleTemp = [];
        foreach ($currentSchedule as $iGame => $game) {
            $currentScheduleTemp[$iGame] = [
                'games' => $game,
                'entposcount' => array_fill(0, $this->entitiesPerGame, 0),
            ];
        }

        $newSchedule = [];
        foreach ($currentScheduleTemp as $iGame => $game) {
            $currentScheduleTemp[$iGame]['entposcount'] = $this->calcEntityTeamSlotCountData($currentScheduleTemp, $iGame);
            $currentScheduleTemp[$iGame]['eplpdist'] = $this->calcDistFromLastPlayedByGameIdx($currentScheduleTemp, $iGame);

            $newSchedule[] = $this->sortGameByEntityPos($currentScheduleTemp[$iGame]);
            $currentScheduleTemp[$iGame]['games'] = $newSchedule[count($newSchedule) - 1];
            $currentScheduleTemp[$iGame]['entposcount'] = [];
            $currentScheduleTemp[$iGame]['eplpdist'] = [];
        }

        return $newSchedule;
    }

    private function sortGameByEntityPos(array $gameWithEntPosCount): array
    {
        $bGTTwoGames = false;
        foreach ($gameWithEntPosCount['entposcount'] as $entPosCount) {
            if (max($entPosCount) >= 2) {
                $bGTTwoGames = true;
                break;
            }
        }

        if (!$bGTTwoGames) {
            return $gameWithEntPosCount['games'];
        }

        $entityCount = count($gameWithEntPosCount['games']);
        $gameEntitiesSorted = [];

        while (count($gameEntitiesSorted) < $entityCount) {
            $arrSort = [];
            if (count($gameWithEntPosCount['games']) == 1) {
                $gameEntitiesSorted[] = $gameWithEntPosCount['games'][0];
                break;
            }

            $entSlot = count($gameEntitiesSorted);
            foreach ($gameWithEntPosCount['entposcount'] as $ipos => $entPosCount) {
                if (isset($gameWithEntPosCount['games'][$ipos])) {
                    $arrSort[] = [
                        'entity' => $gameWithEntPosCount['games'][$ipos],
                        'poscount' => $entPosCount[$entSlot] - (0.01 * $gameWithEntPosCount['eplpdist'][$ipos][$entSlot]),
                        'currpos' => $ipos,
                    ];
                }
            }

            usort($arrSort, fn($a, $b) => $a['poscount'] <=> $b['poscount']);
            $nextEntity = array_shift($arrSort);
            $nextEntityOldIdxPos = $nextEntity['currpos'];

            unset($gameWithEntPosCount['games'][$nextEntityOldIdxPos]);
            unset($gameWithEntPosCount['entposcount'][$nextEntityOldIdxPos]);

            $gameEntitiesSorted[] = $nextEntity['entity'];
        }

        return $gameEntitiesSorted;
    }

    private function calcEntityTeamSlotCountData(array $currentSchedule, int $gameIdx, array $entCountData = [], int $evalTeamSlot = 0): array
    {
        $game = $currentSchedule[$gameIdx]['games'];
        if ($evalTeamSlot > $this->entitiesPerGame - 1) {
            return $entCountData;
        }

        $entCountData[$evalTeamSlot] = array_fill(0, $this->entitiesPerGame, 0);
        $entityToSearchFor = $game[$evalTeamSlot];

        for ($iGame = $gameIdx; $iGame >= 0; $iGame--) {
            for ($iTeamSlot = 0; $iTeamSlot < $this->entitiesPerGame; $iTeamSlot++) {
                if ($currentSchedule[$iGame]['games'][$iTeamSlot] == $entityToSearchFor) {
                    $entCountData[$evalTeamSlot][$iTeamSlot]++;
                }
            }
        }

        return $this->calcEntityTeamSlotCountData($currentSchedule, $gameIdx, $entCountData, $evalTeamSlot + 1);
    }

    private function calcDistFromLastPlayedByGameIdx(array $schedule, int $gameIdx, int $defaultValue = RAW_GAME_ARRAY_COUNT, array $arrDist = [], int $teamSlot = 0): array
    {
        if ($teamSlot > $this->entitiesPerGame - 1) {
            return $arrDist;
        }

        $entityToSearchFor = $schedule[$gameIdx]['games'][$teamSlot];
        $arrDistLocal = array_fill(0, $this->entitiesPerGame, -1);

        for ($iGame = $gameIdx - 1; $iGame >= 0; $iGame--) {
            for ($iTeamSlot = 0; $iTeamSlot < $this->entitiesPerGame; $iTeamSlot++) {
                if ($arrDistLocal[$iTeamSlot] == -1 && $entityToSearchFor == $schedule[$iGame]['games'][$iTeamSlot]) {
                    $arrDistLocal[$iTeamSlot] = $gameIdx - $iGame;
                    break;
                }
            }
        }

        $arrDist[$teamSlot] = $arrDistLocal;
        return $this->calcDistFromLastPlayedByGameIdx($schedule, $gameIdx, $defaultValue, $arrDist, $teamSlot + 1);
    }

    private function calcEntityPrevPlayedModifier(array $currentSchedule, array $game): float
    {
        $ret = 0;
        $modifier = count($currentSchedule);

        foreach ($currentSchedule as $scheduledGame) {
            $similarEntityCount = count($game) - 1 - count(array_diff($game, $scheduledGame));
            if ($similarEntityCount < 0) {
                $similarEntityCount = 0;
            }
            $ret -= $modifier * $similarEntityCount;
        }

        return $ret;
    }

    private function calcGamesPlayed(array $currentSchedule, array $game, array $countdata = [], int $teamSlot = 0): array
    {
        if ($teamSlot > $this->entitiesPerGame - 1) {
            return $countdata;
        }

        $entityToSearchFor = $game[$teamSlot];
        $count = 0;

        for ($iGame = count($currentSchedule) - 1; $iGame >= 0; $iGame--) {
            if (in_array($entityToSearchFor, $currentSchedule[$iGame])) {
                $count++;
            }
        }

        $countdata[$teamSlot] = $count;
        return $this->calcGamesPlayed($currentSchedule, $game, $countdata, $teamSlot + 1);
    }

    public function getGamesPlayedCount(int $entity, ?array $currentSchedule = null): int
    {
        if ($currentSchedule === null) {
            $currentSchedule = $this->schedule;
        }
        $arr = $this->calcGamesPlayed($currentSchedule, [$entity]);
        return $arr[0];
    }

    private function nextGame(array $currentSchedule, array &$remainingGames): ?array
    {
        if (empty($remainingGames)) {
            return null;
        }

        foreach ($remainingGames as &$game) {
            $game['vdata'] = $this->calcDistFromLastPlayed($currentSchedule, $game['games']);
            
            $entdist = 0;
            for ($g = 1; $g < count($game['games']); $g++) {
                $entdist += $game['games'][$g] - $game['games'][$g - 1];
            }
            $game['entdist'] = $entdist / (count($game['games']) - 1);
            
            $game['value'] = array_sum($game['vdata']) + 
                             $this->calcEntityPrevPlayedModifier($currentSchedule, $game['games']);
            
            $game['countdata'] = $this->calcGamesPlayed($currentSchedule, $game['games']);
        }

        if ($this->gamesForEachEntity > 0) {
            $this->removeGamesBeyondMaxGameCount($remainingGames, $this->gamesForEachEntity);
        }

        usort($remainingGames, fn($a, $b) => $a['value'] <=> $b['value']);
        $retGame = array_pop($remainingGames);
        return $retGame['games'];
    }

    private function removeGamesBeyondMaxGameCount(array &$remainingGames, int $maxGameCount = 0): int
    {
        $removedCount = 0;
        if ($maxGameCount == 0) {
            $maxGameCount = $this->gamesForEachEntity;
        }
        if ($maxGameCount == 0 || empty($remainingGames)) {
            return $removedCount;
        }

        usort($remainingGames, fn($a, $b) => max($b['countdata']) <=> max($a['countdata']));

        for ($i = count($remainingGames) - 1; $i >= 0; $i--) {
            if (max($remainingGames[$i]['countdata']) >= $this->gamesForEachEntity) {
                unset($remainingGames[$i]);
                $removedCount++;
            }
        }

        $remainingGames = array_values($remainingGames);
        return $removedCount;
    }

    private function calcRawSchedule(): array
    {
        $rawGamesWork = [];
        foreach ($this->rawGames as $game) {
            $rawGamesWork[] = [
                'games' => $game,
                'value' => 0,
                'vdata' => null,
                'countdata' => null,
                'entdist' => null,
            ];
        }

        $scheduleSorted = [];
        $nextGame = $rawGamesWork[0]['games'];
        array_shift($rawGamesWork);

        do {
            $scheduleSorted[] = $nextGame;
            $nextGame = $this->nextGame($scheduleSorted, $rawGamesWork);
        } while ($nextGame !== null);

        return $scheduleSorted;
    }

private function incGame(array $currentGame, int $currentEntityColumn = -1): ?array
    {
        if ($currentEntityColumn == -1) {
            $currentEntityColumn = $this->entitiesPerGame;
        }
        if ($currentEntityColumn == 0) {
            return null;
        }
        if ($currentGame[$currentEntityColumn - 1] == $this->entityCount) {
            if ($currentEntityColumn >= 1) {
                $currentGame = $this->incGame($currentGame, $currentEntityColumn - 1);
                if ($currentGame === null) {
                    return null;
                }
                $currentGame[$currentEntityColumn - 1] = 1;
            } else {
                return null;
            }
        }
        $currentGame[$currentEntityColumn - 1]++;
        return $currentGame;
    }

    private function createRawGames(): int
    {
        $currentGame = array_fill(0, $this->entitiesPerGame, 1);
        $this->rawGames = [];

        do {
            if (count(array_unique($currentGame)) == $this->entitiesPerGame) {
                $this->rawGames[] = $currentGame;
            }
            $currentGame = $this->incGame($currentGame);
        } while ($currentGame !== null);

        foreach ($this->rawGames as &$game) {
            sort($game);
        }

        usort($this->rawGames, [$this, 'compareGames']);

        $this->rawGames = $this->superUnique($this->rawGames);
        $this->rawGames = array_values($this->rawGames);

        $rawListMultiplier = 1;
        if ($this->gamesForEachEntity > 0) {
            $rawListMultiplier = (int) ceil($this->gamesForEachEntity / ($this->entityCount - 1));
        } elseif ($this->gamesForEachEntity < 0 && $this->gamesForEachEntity >= -4) {
            $rawListMultiplier = abs($this->gamesForEachEntity);
        }

        if ($rawListMultiplier > 1) {
            $rawGamesImage = $this->rawGames;
            for ($i = 2; $i <= $rawListMultiplier; $i++) {
                $this->rawGames = array_merge($this->rawGames, $rawGamesImage);
            }
        }

        return count($this->rawGames);
    }

    public function allGamesPlayed(): bool
    {
        foreach ($this->gamesPlayed as $gamesPlayed) {
            if ($gamesPlayed < $this->gamesForEachEntity) {
                return false;
            }
        }
        return true;
    }

    private function compareGames(array $a, array $b): int
    {
        $l = $this->gameToString($a);
        $r = $this->gameToString($b);
        return $l <=> $r;
    }

    private function gameToString(array $game): string
    {
        return implode('', array_map(fn($entity) => str_pad((string)$entity, 4, '0', STR_PAD_LEFT), $game));
    }

    private function superUnique(array $array): array
    {
        $result = array_map("unserialize", array_unique(array_map("serialize", $array)));

        foreach ($result as $key => $value) {
            if (is_array($value)) {
                $result[$key] = $this->superUnique($value);
            }
        }

        return $result;
    }

    private function writeString(string $fileName, string $stringToWrite, bool $append = true): void
    {
        $mode = $append ? 'a' : 'w';
        $fh = fopen($fileName, $mode);
        if ($fh === false) {
            throw new RuntimeException("Can't open file: $fileName");
        }
        fwrite($fh, $stringToWrite);
        fclose($fh);
    }

    public function dumpGames(string $fileName, array $gamesArrayToDump, bool $newFile = false): void
    {
        if ($newFile && file_exists($fileName)) {
            unlink($fileName);
        }

        $this->writeString($fileName, "Game Dump\n", true);

        foreach ($gamesArrayToDump as $cnt => $game) {
            $str = "[" . str_pad((string)$cnt, 3, '0', STR_PAD_LEFT) . "]:\n";

            if (isset($game['value'])) {
                $str .= "      value:" . $game['value'];
            }

            $displayed = false;

            if (isset($game['games'])) {
                $str .= "\n      games:";
                $str .= implode(',', array_map(fn($entity) => ' ' . str_pad((string)$entity, 2, '0', STR_PAD_LEFT), $game['games']));
                $displayed = true;
            }

            if (isset($game['vdata'])) {
                $str .= "\n      vdata:";
                $str .= implode(',', array_map(fn($value) => ' ' . str_pad((string)$value, 2, '0', STR_PAD_LEFT), $game['vdata']));
                $displayed = true;
            }

            if (isset($game['countdata'])) {
                $str .= "\n  countdata:";
                $str .= implode(',', array_map(fn($count) => ' ' . str_pad((string)$count, 2, '0', STR_PAD_LEFT), $game['countdata']));
                $displayed = true;
            }

            if (isset($game['entposcount'])) {
                $str .= "\n  entposcount: ( ";
                foreach ($game['entposcount'] as $entposcountData) {
                    if (is_array($entposcountData)) {
                        $str .= "[" . implode(',', array_map(fn($count) => ' ' . str_pad((string)$count, 2, '0', STR_PAD_LEFT), $entposcountData)) . "],";
                    }
                }
                $str = rtrim($str, ',');
                $displayed = true;
            }

            if (isset($game['eplpdist'])) {
                $str .= "\n  eplpdist: ( ";
                foreach ($game['eplpdist'] as $eplpDispData) {
                    if (is_array($eplpDispData)) {
                        $str .= "[" . implode(',', array_map(fn($disp) => ' ' . str_pad((string)$disp, 2, '0', STR_PAD_LEFT), $eplpDispData)) . "],";
                    }
                }
                $str = rtrim($str, ',');
                $displayed = true;
            }

            if (!$displayed) {
                $str .= implode(',', $game);
            }

            $str .= "\n";
            $this->writeString($fileName, $str);
        }
    }

    // Getter and setter methods
    public function setError(?string $error): void
    {
        $this->error = $error;
    }

    public function getError(): ?string
    {
        return $this->error;
    }

    public function setEntity(?string $entity): void
    {
        $this->entity = $entity;
    }

    public function getEntity(): ?string
    {
        return $this->entity;
    }

    public function setEntityCount(int $entityCount): void
    {
        $this->entityCount = $entityCount;
    }

    public function getEntityCount(): int
    {
        return $this->entityCount;
    }

    public function setEntitiesPerGame(int $entitiesPerGame): void
    {
        $this->entitiesPerGame = $entitiesPerGame;
    }

    public function getEntitiesPerGame(): int
    {
        return $this->entitiesPerGame;
    }

    public function setSubGamePerGame(int $subGamePerGame): void
    {
        $this->subGamePerGame = $subGamePerGame;
    }

    public function getSubGamePerGame(): int
    {
        return $this->subGamePerGame;
    }

    public function setGamesForEachEntity(int $gamesForEachEntity): void
    {
        $this->gamesForEachEntity = $gamesForEachEntity;
    }

    public function getGamesForEachEntity(): int
    {
        return $this->gamesForEachEntity;
    }
}